<?php
function validateRequestProcess($validationRules,$data){
        $errors = [];
    
    foreach ($validationRules as $field => $rule) {
        // Verificar si el campo es obligatorio
        if ($rule['required'] && !isset($data[$field])) {
            $errors[] = "El campo '$field' es requerido";
            continue;
        }
        // Si el campo no está presente y no es obligatorio, continuar
        if (!isset($data[$field]) && !$rule['required']) {
            continue;
        }
        // Verificar si el campo puede estar vacio
        if ($rule['notEmpty'] && $data[$field]==""){
            $errors[] = "El campo '$field' no puede actualizarse a vacio";
            continue;
        }
        
        if(isset($rule['isStatus']) && !($data[$field] === "ACTIVO" || $data[$field] === "INACTIVO" || $data[$field] === "BLOQUEADO")){
            $errors[] = "El campo '$field' solo acepta los valores ACTIVO, INACTIVO o BLOQUEADO";
            continue;
        }

        $value = $data[$field];
        
        // Validar tipo de dato
        switch ($rule['type']) {
            case 'string':
                if (!is_string($value)) $errors[] = "El campo '$field' debe ser un string";
                if (isset($rule['max_length']) && strlen($value) > $rule['max_length']) $errors[] = "El campo '$field' excede el largo máximo de {$rule['max_length']}";
                if (isset($rule['length']) && strlen($value) != $rule['length']) $errors[] = "El campo '$field' debe tener un largo de {$rule['length']}";
                break;
            case 'integer':
                if (!is_numeric($value) || floor($value) != $value) $errors[] = "El campo '$field' debe ser un integer";
                if (isset($rule['min']) && $value < $rule['min']) $errors[] = "El campo '$field' no puede tener un valor inferior a {$rule['min']}";
                if (isset($rule['max']) && $value > $rule['max']) $errors[] = "El campo '$field' no puede tener un valor mayor a {$rule['max']}";
                break;
            case 'email':
                if (!filter_var($value, FILTER_VALIDATE_EMAIL)) $errors[] = "El campo '$field' debe ser un email valido";
                break;
            case 'boolean':
                if (!is_bool($value) && $value !== 0 && $value !== 1) $errors[] = "El campo '$field' debe ser un booleano";
                break;
            case 'datetime':
                $format = 'Y-m-d H:i:s';
                $dateTime = DateTime::createFromFormat($format, $value);
                if (!$dateTime || $dateTime->format($format) !== $value) $errors[] = "El campo '$field' debe contener una fecha valida (YYYY-MM-DD HH:MM:SS)";
                break;
            case 'date':
                $format = 'Y-m-d';
                $dateTime = DateTime::createFromFormat($format, $value);
                if (!$dateTime || $dateTime->format($format) !== $value) $errors[] = "El campo '$field' debe contener una fecha valida (YYYY-MM-DD)";
                break;
            case 'field':
                if($value['type'] != 'text/csv') $errors[] = "El archivo ".$value['name']." debe tener un formato valido";
                if($value['size'] > 5242880) $errors[] = "El archivo ".$value['name']." excede el peso maximo";
                break;
        }
    
    }
    return $errors;
}
class ValidateRequest{
    public static function validateData($data){
        $validationRules = [
            'tipo-organizacion' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => true
            ],
            'tipo-plantilla' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => true
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateField($data){
        $validationRules = [
            'plantilla' => [
                'type' => 'field',
                'notEmpty' => true,
                'required' => true
            ]
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateGetSubdimensionField($data){
        $validationRules = [
            'rol' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => true
            ],
            'dimension' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => true
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateUpdatePlantillaField($data){
        $validationRules = [
            'nombre' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false
            ],
            'descripcion' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false
            ],
            'estado' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false,
                'isStatus' => true
            ],
            'id_tipo_organizacion' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => false
            ],
            'id_tipo_plantilla' => [
                'type' => 'integer',
                'notEmpty' => true,
                'required' => false
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateUpdateDimensionField($data){
        $validationRules = [
            'nombre' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateUpdateSubDimensionField($data){
        $validationRules = [
            'nombre' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
    public static function validateUpdateIndicadorDesempenoField($data){
        $validationRules = [
            'nombre' => [
                'type' => 'string',
                'notEmpty' => true,
                'required' => false
            ],
        ];
        $errors = validateRequestProcess($validationRules,$data);
        return $errors;
    }
}
?>